/*
 * routines for loading the fabric description from a file
 */
#include <stdio.h>

#include "libfma.h"
#include "lf_db.h"
#include "lf_internal.h"
#include "lf_fabric.h"

#define _LF_DEFINE_TABLES_
#include "lf_fabric_db.h"

#include "lf_product_def.h"
#include "lf_switch.h"

static int lf_fill_linecard(lf_linecard_t *lp);
static void lf_make_linecard_edge_links(lf_linecard_t *lp);

/*
 * given a database pointer, load up the whole fabric
 */
struct lf_fabric *
lf_load_fabric(
  struct lf_fabric_db *fdp)
{
  struct lf_fabric *fp;
  int rc;

  /* setup variables */
  fp = NULL;

  /* allocate holder for fabric */
  LF_CALLOC(fp, struct lf_fabric, 1);

  /* load up all the hosts.  No hosts is not an error. */
  rc = lf_load_hosts(fdp, fp);
  if (rc != 0) goto except;

  /* load up all the nics.  No nics is not an error. */
  rc = lf_load_nics(fdp, fp);
  if (rc != 0) goto except;

  /* load up all the enclosures.  No enclosures is not an error. */
  rc = lf_load_enclosures(fdp, fp);
  if (rc != 0) goto except;

  /* If we found enclosures, load up their linecard information */
  if (fp->enclosures != NULL) {
    rc = lf_load_linecards(fdp, fp);
    if (rc != 0) goto except;
  }

  /* load linkage information */
  rc = lf_load_links(fdp, fp);
  if (rc != 0) goto except;

  /* If DB needs rewriting, do it now */
  if (fdp->db_needs_rewrite) {
    rc = lf_rewrite_fabric_db(fdp, fp);
    if (rc != 0) LF_ERROR(("Error rewriting database, try fm_fixup_db"));
  }

  return fp;

 except:
  if (fp != NULL) lf_free_fabric(fp);
  return NULL;
}

/*
 * free a loaded fabric
 */
void
lf_free_fabric(
  lf_fabric_t *fp)
{
  int i;

  if (fp == NULL) return;

  /* call fabric user destructor if specified and needed */
  if (fp->user.v != NULL && fp->fabric_user_destructor != NULL) {
    fp->fabric_user_destructor(fp);
  }

  /* free the hosts */
  for (i=0; i<fp->num_hosts; ++i) {
    lf_free_host(fp->hosts[i]);
  }
  LF_FREE(fp->hosts);

  /* Free xbars via enclosures if there are any */
  if (fp->num_enclosures != 0) {

    /* free the enclosures */
    for (i=0; i<fp->num_enclosures; ++i) {
      lf_free_enclosure(fp->enclosures[i]);
    }
    LF_FREE(fp->enclosures);

  /* If there are no enclosures, try freeing the xbars from the xbar array */
  } else if (fp->num_xbars > 0) {

    /* Free the xbars this way */
    for (i=0; i<fp->num_xbars; ++i) {
      lf_free_xbar(fp->xbars[i]);
    }
  }

  LF_FREE(fp->xbars);		/* xbars shortcut array */

  LF_FREE(fp->lag_ids);		/* free LAG_ID array */

  /* free the fabric struct itself */
  LF_FREE(fp);
}

void
lf_free_host(
  lf_host_t *hp)
{
  int n;

  if (hp == NULL) return;

  /* call host user destructor if specified and needed */
  if (hp->user.v != NULL && hp->host_user_destructor != NULL) {
    hp->host_user_destructor(hp);
  }

  /* free the NICs associated with this host */
  for (n=0; n<hp->num_nics; ++n) {
    lf_free_nic(hp->nics[n]);
  }
  LF_FREE(hp->hostname);
  LF_FREE(hp->nics);
  LF_FREE(hp);
}

/*
 * Allocate a NIC and it's port-based subarrays
 */
struct lf_nic *
lf_alloc_nic(
  int num_ports)
{
  struct lf_nic *nicp;
  int p;

  LF_CALLOC(nicp, struct lf_nic, 1);
  nicp->ln_type = LF_NODE_NIC;
  nicp->num_ports = num_ports;

  /* alloacte empty arrays of topo and phys pointers */
  LF_CALLOC(nicp->phys_ports, union lf_node *, nicp->num_ports);
  LF_CALLOC(nicp->phys_rports, int, nicp->num_ports);
  LF_CALLOC(nicp->topo_ports, union lf_node *, nicp->num_ports);
  LF_CALLOC(nicp->topo_rports, int, nicp->num_ports);
  LF_CALLOC(nicp->link_state, enum lf_link_state, nicp->num_ports);
  LF_CALLOC(nicp->link_topo_index, int, nicp->num_ports);


  /* default the state of each link to up */
  for (p=0; p<num_ports; ++p) {
    nicp->link_state[p] = LF_LINK_STATE_UP;
  }

  return nicp;

 except:
  lf_free_nic(nicp);
  return NULL;
}

/*
 * Free an xbar and all of its parts.  user struct should already be freed
 */
void
lf_free_nic(
  lf_nic_t *nicp)
{
  int i;

  if (nicp == NULL) return;

  /* call nic user destructor if specified and needed */
  if (nicp->user.v != NULL && nicp->nic_user_destructor != NULL) {
    nicp->nic_user_destructor(nicp);
  }

  /* free the xcvrs associated with this NIC */
  if (nicp->phys_ports != NULL) {
    for (i=0; i<nicp->num_ports; ++i) {
      union lf_node *np;

      np = nicp->phys_ports[i];
      if (np != NULL && np->ln_type == LF_NODE_NIC_XCVR) {
	lf_free_xcvr(LF_XCVR(np));
      }
    }
  }
  LF_FREE(nicp->serial_no);
  LF_FREE(nicp->product_id);
  LF_FREE(nicp->topo_ports);
  LF_FREE(nicp->topo_rports);
  LF_FREE(nicp->phys_ports);
  LF_FREE(nicp->phys_rports);
  LF_FREE(nicp->link_state);
  LF_FREE(nicp->link_topo_index);

  if (nicp->data != NULL) {
    for (i=0; i>nicp->num_ports; ++i) {
      LF_FREE(nicp->data[i]);
    }
  }
  LF_FREE(nicp->data);

  LF_FREE(nicp);
}

void
lf_free_node(
  lf_node_t *np)
{
  if (np == NULL) return;
  switch (np->ln_type) {
  case LF_NODE_NIC_XCVR:
  case LF_NODE_LC_XCVR:
    lf_free_xcvr(LF_XCVR(np));
    break;

  case LF_NODE_NIC:
    lf_free_nic(LF_NIC(np));
    break;

  case LF_NODE_XBAR:
    lf_free_xbar(LF_XBAR(np));
    break;
  }
}

/*
 * Allocate an xbar and it's port-based subarrays
 */
struct lf_xbar *
lf_alloc_xbar(
  int num_ports)
{
  struct lf_xbar *xp;
  int p;

  LF_CALLOC(xp, struct lf_xbar, 1);
  xp->num_ports = num_ports;
  xp->ln_type = LF_NODE_XBAR;

  LF_CALLOC(xp->phys_ports, union lf_node *, xp->num_ports);
  LF_CALLOC(xp->phys_rports, int, xp->num_ports);
  LF_CALLOC(xp->topo_ports, union lf_node *, xp->num_ports);
  LF_CALLOC(xp->topo_rports, int, xp->num_ports);
  LF_CALLOC(xp->link_state, enum lf_link_state, xp->num_ports);
  LF_CALLOC(xp->link_topo_index, int, xp->num_ports);

  /* default the state of each link to up */
  for (p=0; p<num_ports; ++p) {
    xp->link_state[p] = LF_LINK_STATE_UP;
  }

  LF_CALLOC(xp->verifiers, struct lf_verifier, xp->num_ports);
  LF_CALLOC(xp->xbar_data, struct lf_xbar_data, 1);
  LF_CALLOC(xp->data, struct lf_xbarport_data, xp->num_ports);
  return xp;

 except:
  lf_free_xbar(xp);
  return NULL;
}

/*
 * Free an xbar and all of its parts.  user struct should already be freed
 */
void
lf_free_xbar(
  lf_xbar_t *xp)
{
  if (xp == NULL) return;

  /* call xbar user destructor if specified and needed */
  if (xp->user.v != NULL && xp->xbar_user_destructor != NULL) {
    xp->xbar_user_destructor(xp);
  }

  LF_FREE(xp->topo_ports);
  LF_FREE(xp->topo_rports);
  LF_FREE(xp->phys_ports);
  LF_FREE(xp->phys_rports);
  LF_FREE(xp->link_state);
  LF_FREE(xp->link_topo_index);
  LF_FREE(xp->verifiers);
  LF_FREE(xp->xbar_data);
  LF_FREE(xp->data);
  LF_FREE(xp);
}

void
lf_free_xcvr(
  lf_xcvr_t *xcp)
{
  if (xcp == NULL) return;

  /* call xcvr user destructor if specified and needed */
  if (xcp->user.v != NULL && xcp->xcvr_user_destructor != NULL) {
    xcp->xcvr_user_destructor(xcp);
  }

  LF_FREE(xcp->ports);
  LF_FREE(xcp->rports);
  LF_FREE(xcp->data);
  LF_FREE(xcp);
}

void
lf_free_linecard(
  lf_linecard_t *lp)
{
  int x;

  if (lp == NULL) return;

  /* call linecard user destructor if specified and needed */
  if (lp->user.v != NULL && lp->linecard_user_destructor != NULL) {
    lp->linecard_user_destructor(lp);
  }

  /* free strings and arrays */
  LF_FREE(lp->serial_no);
  LF_FREE(lp->product_id);
  LF_FREE(lp->xcvr_labels);
  LF_FREE(lp->data);

  /* free all the xbars */
  for (x=0; x<lp->num_xbars; ++x) {
    lf_free_xbar(LF_XBAR(lp->xbars[x]));
  }
  LF_FREE(lp->xbars);

  /* free all the xcvrs */
  for (x=0; x<lp->num_xcvrs; ++x) {
    lf_free_xcvr(LF_XCVR(lp->xcvrs[x]));
  }
  LF_FREE(lp->xcvrs);

  LF_FREE(lp);
}

void
lf_free_enclosure(
  lf_enclosure_t *ep)
{
  int l;

  if (ep == NULL) return;

  /* call enclosure user destructor if specified and needed */
  if (ep->user.v != NULL && ep->enclosure_user_destructor != NULL) {
    ep->enclosure_user_destructor(ep);
  }

  LF_FREE(ep->name);
  LF_FREE(ep->product_id);
  LF_FREE(ep->data);

  /* free all the linecards */
  for (l=0; l<ep->num_slots; ++l) {
    if (ep->slots[l] != NULL) {
      lf_free_linecard(ep->slots[l]);
    }
  }
  LF_FREE(ep->slots);

  LF_FREE(ep);
}

/*
 * Load the hosts database into internal structures
 */
int
lf_load_hosts(
  struct lf_fabric_db *fdp,
  struct lf_fabric *fp)
{
  int ret;
  db_table_ptr_t tp;
  db_datum_t criteria[2];
  db_datum_t **hdp;
  struct lf_host **hp;
  int rc;
  int nh;
  int i;
  int *colid;

  /* init vars */
  hp = NULL;
  hdp = NULL;
  tp = NULL;

  /* open the hosts table */
  colid = fdp->host_colids;
  if (fdp->host_tbl == NULL) {
    tp = db_open_table(fdp->database, LF_TABLE_HOSTS);
    if (tp == NULL) goto no_hosts_ok;
    fdp->host_tbl = tp;

    /* get the column indices */
    rc = lf_get_column_indices(tp,
                             lf_table_host_cols,
			     LF_TABLE_HOST_NUMCOLS,
			     colid,
			     LF_TABLE_HOSTS);
    if (rc == -1) goto except;
  } else {
    tp = fdp->host_tbl;
  }

  /* Check for missing columns */
  for (i=0; i<LF_TABLE_HOST_NUMCOLS; ++i) {
    if (colid[i] == -1) {
      fdp->db_needs_rewrite = TRUE;
    }
  }

  /* Build criteria for the DB query */
  criteria[0].dd_type = DBD_INT;
  criteria[0].dbd_int = LF_DS_CURRENT;
  criteria[1].dd_type = DBD_INT;
  criteria[1].dbd_int = LF_DS_NEW;

  /* get all hosts - none is not an error */
  rc = db_simple_or_query(tp, colid[LF_TABLE_HOST_STATUS],
                          criteria, 2, &hdp, &nh);
  if (rc == -1) goto except;
  if (hdp == NULL) goto no_hosts_ok;

  /* allocate a place for the hosts */
  LF_CALLOC(hp, struct lf_host *, nh);
  fp->hosts = hp;
  fp->num_hosts = nh;

  /* fill in the host structures */
  for (i=0; i<nh; ++i) {
    LF_CALLOC(hp[i], struct lf_host, 1);
    LF_DUP_STRING(hp[i]->hostname,
		  hdp[i][colid[LF_TABLE_HOST_HOSTNAME]].dbd_str);
    hp[i]->fw_type = hdp[i][colid[LF_TABLE_HOST_FW_TYPE]].dbd_int;
    hp[i]->fma_flags = hdp[i][colid[LF_TABLE_HOST_FLAGS]].dbd_int;
    hp[i]->db_status = hdp[i][colid[LF_TABLE_HOST_STATUS]].dbd_int;
  }

  /* free the query results */
  db_free_query_res(tp, hdp, nh);

  /* everything looks ok, return the data */
  return 0;

 no_hosts_ok:
  ret = 0;
  goto cleanup;

 except:
  ret = -1;
 cleanup:
  if (hdp != NULL) db_free_query_res(tp, hdp, nh);
  return ret;
}

/*
 * Allocate a NIC based on product ID
 * num_ports may be specified, or may be -1.  If not -1, and the NIC
 * productinfo is not found, then treat it as a generic NIC
 */
struct lf_nic *
lf_alloc_nic_by_product_id(
  char *product_id)
{
  struct lf_nic_def *dp;
  struct lf_nic *np;
  int p;

  dp = lf_get_nic_definition(product_id);
  if (dp == NULL) {
    LF_ERROR(("Loading NIC definition for %s", product_id));
  }

  /* allocate the NIC struct */
  np = lf_alloc_nic(dp->num_links);
  if (np == NULL) LF_ERROR(("Error allocating NIC"));

  np->def = dp;	/* save product definition */
  LF_DUP_STRING(np->product_id, product_id);

  /* fill in array of transceivers */
  if (dp->num_xcvrs != 0 && dp->num_xcvrs != np->num_ports) {
    LF_ERROR(("Weird number of xcvrs for %s", dp->product_id));
  }

  /* allocate and fill in the transceivers */
  for (p=0; p<dp->num_xcvrs; ++p) {
    struct lf_xcvr *nxp;

    LF_CALLOC(nxp, struct lf_xcvr, 1);

    nxp->ln_type = LF_NODE_NIC_XCVR;
    nxp->p.nic = np;
    nxp->port = p;

    /* assume only single port transceiver */
    nxp->num_conns = np->def->xcvr_ports;
    nxp->num_ports = nxp->num_conns * 2;
    LF_CALLOC(nxp->ports, lf_node_t *, nxp->num_ports);
    LF_CALLOC(nxp->rports, int, nxp->num_ports);

    /* link NIC to port 1; port 0 is external link */
    lf_make_phys_link(LF_NODE(np), p, LF_NODE(nxp), 1);
    lf_make_phys_link(LF_NODE(nxp), 1, LF_NODE(np), p);
  }
  return np;

 except:
  return NULL;
}

/*
 * Load the NICs given a fabric
 */
int
lf_load_nics(
  struct lf_fabric_db *fdp,
  struct lf_fabric *fp)
{
  int rc;
  int retval;
  db_table_ptr_t tp;
  db_datum_t **ndp;
  db_datum_t **criteria;
  int n_nics;
  int *colid;
  int i;
  int j;

  /* init vars */
  tp = NULL;
  ndp = NULL;
  retval = 0;
  criteria = NULL;

  /* open the NICs table */
  colid = fdp->nic_colids;
  if (fdp->nic_tbl == NULL) {
    tp = db_open_table(fdp->database, LF_TABLE_NICS);
    if (tp == NULL) goto finish;
    fdp->nic_tbl = tp;

    /* get the column indices */
    rc = lf_get_column_indices(tp,
			       lf_table_nic_cols,
			       LF_TABLE_NIC_NUMCOLS,
			       colid,
			       "nics");
    if (rc == -1) LF_ERROR(("NIC table is corrupt"));
  } else {
    tp = fdp->nic_tbl;
  }

  /* Check for missing columns */
  for (j=0; j<LF_TABLE_NIC_NUMCOLS; ++j) {
    if (colid[j] == -1) {
      fdp->db_needs_rewrite = TRUE;
    }
  }

  /* Build a 2-row OR query looking for NEW or CURRENT records */
  LF_CALLOC(criteria, db_datum_t *, 2);
  for (i=0; i<2; ++i) {
    LF_CALLOC(criteria[i], db_datum_t, LF_TABLE_NIC_NUMCOLS);
    for (j=0; j<LF_TABLE_NIC_NUMCOLS; ++j) {
      criteria[i][j].dd_type = DBD_MATCHALL;
    }
    criteria[i][colid[LF_TABLE_NIC_HOSTNAME]].dd_type = DBD_STR;
    criteria[i][colid[LF_TABLE_NIC_STATUS]].dd_type = DBD_INT;
  }
  criteria[0][colid[LF_TABLE_NIC_STATUS]].dbd_int = LF_DS_CURRENT;
  criteria[1][colid[LF_TABLE_NIC_STATUS]].dbd_int = LF_DS_NEW;

  /* fill in the nic information for each host */
  for (i=0; i<fp->num_hosts; ++i) {
    struct lf_host *hp;
    int n;

    hp = fp->hosts[i];

    /* finish the search spec */
    criteria[0][colid[LF_TABLE_NIC_HOSTNAME]].dbd_str = hp->hostname;
    criteria[1][colid[LF_TABLE_NIC_HOSTNAME]].dbd_str = hp->hostname;

    rc = db_row_or_query(tp, criteria, 2, &ndp, &n_nics);
    if (rc == -1) goto except;
    if (ndp == NULL) continue;

    /* allocate space for the NIC info */
    LF_CALLOC(hp->nics, struct lf_nic *, n_nics);
    hp->num_nics = n_nics;

    /* fill in info for each NIC */
    for (n=0; n<n_nics; ++n) {
      struct lf_nic *nicp;
      char *product_id;

      /* get product information on this NIC */
      product_id = ndp[n][colid[LF_TABLE_NIC_PRODUCT_ID]].dbd_str;

      nicp = lf_alloc_nic_by_product_id(product_id);
      if (nicp == NULL) LF_ERROR(("Allocating NIC"));

      nicp->host = hp;	/* save reverse pointer to host */
      nicp->slot = n;
      hp->nics[n] = nicp;

      /* save fields from database */
      nicp->host_nic_id = ndp[n][colid[LF_TABLE_NIC_HOST_NIC_ID]].dbd_int;
      LF_MAC_COPY(nicp->mac_addr, ndp[n][colid[LF_TABLE_NIC_MAC_ADDR]].dbd_mac);
      LF_DUP_STRING(nicp->serial_no,
		    ndp[n][colid[LF_TABLE_NIC_SERIAL_NO]].dbd_str);
      nicp->db_status = ndp[n][colid[LF_TABLE_NIC_STATUS]].dbd_int;

      /* partition is an added colimn */
      if (colid[LF_TABLE_NIC_PARTITION] != -1) {
	nicp->partition = ndp[n][colid[LF_TABLE_NIC_PARTITION]].dbd_int;
      }
    }

    /* release our query result */
    db_free_query_res(tp, ndp, n_nics);
    ndp = NULL;
  }

  goto finish;

 except:
  retval = -1;
  goto finish;

 finish:
  if (ndp != NULL) db_free_query_res(tp, ndp, n_nics);
  if (criteria != NULL) {
    for (i=0; i<2; ++i) {
      LF_FREE(criteria[i]);
    }
    LF_FREE(criteria);
  }
  return retval;
}

/*
 * load the enclosures
 */
int
lf_load_enclosures(
  struct lf_fabric_db *fdp,
  lf_fabric_t *fp)
{
  int i;
  int rc;
  int retval;
  db_table_ptr_t tp;
  db_datum_t criteria[2];
  db_datum_t **edp;
  int ne;
  lf_enclosure_t **ep;
  int *colid;

  /* init vars */
  retval = 0;
  tp = NULL;
  edp = NULL;

  /* open the enclosuress table */
  colid = fdp->enclosure_colids;
  if (fdp->enclosure_tbl == NULL) {
    tp = db_open_table(fdp->database, LF_TABLE_ENCLOSURES);
    if (tp == NULL) goto cleanup;
    fdp->enclosure_tbl = tp;

    /* get the column indices */
    rc = lf_get_column_indices(tp,
			       lf_table_enclosure_cols,
			       LF_TABLE_ENCLOSURE_NUMCOLS,
			       colid,
			       "enclosures");
    if (rc == -1) goto except;
  } else {
    tp = fdp->enclosure_tbl;
  }

  /* Build criteria for the DB query */
  criteria[0].dd_type = DBD_INT;
  criteria[0].dbd_int = LF_DS_CURRENT;
  criteria[1].dd_type = DBD_INT;
  criteria[1].dbd_int = LF_DS_NEW;

  /* get all enclosures - none is not an error */
  rc = db_simple_or_query(tp, colid[LF_TABLE_ENCLOSURE_STATUS],
                          criteria, 2, &edp, &ne);
  if (rc == -1) goto except;
  if (edp == NULL) goto cleanup;

  /* allocate a place for the enclosures */
  LF_CALLOC(ep, struct lf_enclosure *, ne);
  fp->enclosures = ep;
  fp->num_enclosures = ne;

  /* fill in each enclosure */
  for (i=0; i<ne; ++i) {
    char *product_id;
    char *name;

    product_id = edp[i][colid[LF_TABLE_ENCLOSURE_PRODUCT_ID]].dbd_str;
    name = edp[i][colid[LF_TABLE_ENCLOSURE_NAME]].dbd_str;

    ep[i] = lf_allocate_enclosure(product_id, name);
    if (ep[i] == NULL) LF_ERROR(("Allocating enclosure struct for %s", name));

    ep[i]->db_status = edp[i][colid[LF_TABLE_ENCLOSURE_STATUS]].dbd_int;
  }

  /* successful return */
  goto cleanup;

 except:
  retval = -1;
 cleanup:
  if (edp != NULL) db_free_query_res(tp, edp, ne);
  return retval;;
}

/*
 * Fill in information about an enclosure using its product ID
 */
int
lf_fill_enclosure(
  lf_enclosure_t *ep)
{
  struct lf_enclosure_def *dp;

  /* go load the enclosure definition for this */
  dp = lf_get_enclosure_definition(ep->product_id);
  if (dp == NULL) {
    LF_ERROR(("Loading enclosure definition for %s", ep->product_id));
  }

  /* fill in counts and allocate based on definition */
  ep->def = dp;

  /* linecards */
  ep->num_lc_slots = dp->num_lc_slots;
  ep->num_bp_slots = dp->num_bp_slots;
  ep->num_slots = dp->num_lc_slots + dp->num_bp_slots;
  ep->lc_slotbase = dp->lc_slotbase;
  ep->bp_slotbase = dp->bp_slotbase;
  if (ep->num_slots > 0) {
    LF_CALLOC(ep->slots, struct lf_linecard *, ep->num_slots);
  }

  /* backplane cards */
  if (ep->num_bp_slots > 0) {
    int slot;
    struct lf_linecard *lp;

    /* backplane cards are implicitly present - fill them in */
    for (slot=ep->num_lc_slots; slot<ep->num_slots; ++slot) {
      lp = lf_allocate_linecard(ep, slot, dp->bp_product_id, NULL);
      if (lp == NULL) {
	LF_ERROR(("Allocating backplane slot %d for %s", slot, ep->name));
      }
    }
  }

  return 0;

 except:
  return -1;
}

/*
 * load the linecards
 */
int
lf_load_linecards(
  struct lf_fabric_db *fdp,
  lf_fabric_t *fp)
{
  int i;
  int j;
  int rc;
  int retval;
  db_table_ptr_t tp;
  db_datum_t **criteria;
  db_datum_t **ldp;
  int n_lcs;
  int *colid;

  /* init vars */
  retval = 0;
  tp = NULL;
  ldp = NULL;
  criteria = NULL;

  /* open the linecards table */
  colid = fdp->linecard_colids;
  if (fdp->linecard_tbl == NULL) {
    tp = db_open_table(fdp->database, LF_TABLE_LINECARDS);
    if (tp == NULL) goto cleanup;	/* no table is OK */
    fdp->linecard_tbl = tp;

    rc = lf_get_column_indices(tp,
                             lf_table_linecard_cols,
			     LF_TABLE_LINECARD_NUMCOLS,
			     colid,
			     "linecards");
    if (rc == -1) LF_ERROR(("linecard table is corrupt"));
  } else {
    tp = fdp->linecard_tbl;
  }

  /* Build a 2-row OR query looking for NEW or CURRENT records */
  LF_CALLOC(criteria, db_datum_t *, 2);
  for (i=0; i<2; ++i) {
    LF_CALLOC(criteria[i], db_datum_t, LF_TABLE_LINECARD_NUMCOLS);
    for (j=0; j<LF_TABLE_LINECARD_NUMCOLS; ++j) {
      criteria[i][j].dd_type = DBD_MATCHALL;
    }
    criteria[i][colid[LF_TABLE_LINECARD_ENCLOSURE_NAME]].dd_type = DBD_STR;
    criteria[i][colid[LF_TABLE_LINECARD_STATUS]].dd_type = DBD_INT;
  }
  criteria[0][colid[LF_TABLE_LINECARD_STATUS]].dbd_int = LF_DS_CURRENT;
  criteria[1][colid[LF_TABLE_LINECARD_STATUS]].dbd_int = LF_DS_NEW;
           
  /* fill in the linecard information for each enclosure */
  for (i=0; i<fp->num_enclosures; ++i) {
    lf_enclosure_t *ep;
    int n;

    ep = fp->enclosures[i];

    /* finish the search spec */
    criteria[0][colid[LF_TABLE_LINECARD_ENCLOSURE_NAME]].dbd_str = ep->name;
    criteria[1][colid[LF_TABLE_LINECARD_ENCLOSURE_NAME]].dbd_str = ep->name;

    rc = db_row_or_query(tp, criteria, 2, &ldp, &n_lcs);
    if (rc == -1) goto except;
    if (ldp == NULL) continue;

    /* Parse each line card */
    for (n=0; n<n_lcs; ++n) {
      int slot;
      int userslot;
      lf_linecard_t *lp;
      char *product_id;
      char *serial_no;

      /* validate slot number */
      userslot = ldp[n][colid[LF_TABLE_LINECARD_ENCLOSURE_SLOT]].dbd_int;
      slot = userslot - ep->lc_slotbase;
      if (slot < 0 || slot >= ep->num_lc_slots) {
	LF_ERROR(("Slot number %d out of range on enclosure %s\n",
		 userslot, ep->name));
      }

      /* fill in rest of linecard info */
      product_id = ldp[n][colid[LF_TABLE_LINECARD_PRODUCT_ID]].dbd_str;
      serial_no = ldp[n][colid[LF_TABLE_LINECARD_SERIAL_NO]].dbd_str;
      lp = lf_allocate_linecard(ep, slot, product_id, serial_no);
      if (lp == NULL) {
	LF_ERROR(("Allocating slot %d of %s", userslot, ep->name));
      }

      lp->db_status = ldp[n][colid[LF_TABLE_LINECARD_STATUS]].dbd_int;

      if (rc == -1) goto except;
    }

    /* free query results */
    if (ldp != NULL) db_free_query_res(tp, ldp, n_lcs);
    ldp = NULL;
  }

  /* successful return */
  goto cleanup;

 except:
  retval = -1;
 cleanup:
  if (ldp != NULL) db_free_query_res(tp, ldp, n_lcs);
  if (criteria != NULL) {
    for (i=0; i<2; ++i) {
      LF_FREE(criteria[i]);
    }
    LF_FREE(criteria);
  }
  return retval;
}

/*
 * Return the xbar ID for a given xbar on a linecard
 */
int
lf_get_xbar_id(
  struct lf_linecard *lp,
  int xbar_no)
{
  /* generate xbar ID based on style */
  switch (lp->def->xbar_id_style) {
    case LF_XS_SAME:
      return atoi(lp->serial_no);

    case LF_XS_SERIALX16:
      {
	int serial_no;

	serial_no = atoi(lp->serial_no);
	return (serial_no << 4) | xbar_no;
      }
    default:
      return 0;
  }
}

/*
 * Fill in information about an linecard using its product ID
 */
static int
lf_fill_linecard(
  lf_linecard_t *lp)
{
  struct lf_linecard_def *dp;
  int x;

  /* get the linecard definition */
  dp = lf_get_linecard_definition(lp->product_id);
  if (dp == NULL) LF_ERROR(("Cannot load definition for %s", lp->product_id));

  /* copy over linecard info and allocate structs */
  lp->def = dp;
  
  /* xbar info */
  lp->num_xbars = dp->num_xbars;
  if (lp->num_xbars > 0) {
    LF_CALLOC(lp->xbars, union lf_node *, lp->num_xbars);
  }

  /* transceiver info */
  lp->num_xcvrs = dp->num_xcvrs;
  if (lp->num_xcvrs > 0) {
    LF_CALLOC_C(lp->xcvrs, union lf_node **, struct lf_xcvr *, lp->num_xcvrs);
    LF_CALLOC(lp->xcvr_labels, int, lp->num_xcvrs);
  }

  /* fill in each xbar */
  for (x=0; x<lp->num_xbars; ++x) {
    struct lf_xbar *xp;

    xp = lf_alloc_xbar(dp->num_xbar_ports);
    if (xp == NULL) LF_ERROR(("Allocating xbar"));

    lp->xbars[x] = LF_NODE(xp);
    xp->xbar_no = x;
    xp->linecard = lp;

    xp->xbar_id = lf_get_xbar_id(lp, x);
  }

  /* fill in each transceiver */
  for (x=0; x<lp->num_xcvrs; ++x) {
    struct lf_xcvr *xp;

    LF_CALLOC(xp, struct lf_xcvr, 1);
    lp->xcvrs[x] = LF_NODE(xp);
    xp->ln_type = LF_NODE_LC_XCVR;
    xp->port = x;
    xp->p.linecard = lp;

    /* see discussion of transceiver ports in lf_fabric.h */
    xp->num_conns = dp->num_xcvr_ports;
    xp->num_ports = dp->num_xcvr_ports * 2;
    LF_CALLOC(xp->ports, union lf_node *, xp->num_ports);
    LF_CALLOC(xp->rports, int, xp->num_ports);
    LF_CALLOC(xp->data, struct lf_xcvr_data, 1);

    lp->xcvr_labels[x] = dp->xcvr_labels[x];	/* save silkscreen label */
  }


  return 0;
 except:
  return -1;
}

/*
 * load the links
 */
int
lf_load_links(
  struct lf_fabric_db *fdp,
  lf_fabric_t *fp)
{
  db_table_ptr_t tp;
  db_datum_t **ldp;
  db_datum_t criteria[2];
  int nl;
  int rc;
  int i;
  int retval;
  int *colid;

  /* init vars */
  tp = NULL;
  ldp = NULL;
  retval = 0;

  /* Open table of links */
  colid = fdp->link_colids;
  if (fdp->link_tbl == NULL) {
    tp = db_open_table(fdp->database, LF_TABLE_LINKS);
    if (tp == NULL) goto cleanup;

    /* translate column indices */
    rc = lf_get_column_indices(tp,
                             lf_table_link_cols,
			     LF_TABLE_LINK_NUMCOLS,
			     colid,
			     "links");
    if (rc == -1) LF_ERROR(("links table corrupt"));

    fdp->link_tbl = tp;
  } else {
    tp = fdp->link_tbl;
  }
  /* Build criteria for the DB query */
  criteria[0].dd_type = DBD_INT;
  criteria[0].dbd_int = LF_DS_CURRENT;
  criteria[1].dd_type = DBD_INT;
  criteria[1].dbd_int = LF_DS_NEW;

  /* read in the table */
  rc = db_simple_or_query(tp, colid[LF_TABLE_LINK_STATUS],
                          criteria, 2, &ldp, &nl);
  if (rc == -1) goto except;

  /* make each link */
  for (i=0; i<nl; ++i) {
    lf_node_t *n1;
    lf_node_t *n2;
    lf_node_t *topo1;
    lf_node_t *topo2;
    lf_node_t *onp;
    int oport;
    int tport1;
    int tport2;
    int element_label1;
    int element_label2;
    int port1;
    int port2;

    element_label1 = ldp[i][colid[LF_TABLE_LINK_PORT_1]].dbd_int;
    port1 = ldp[i][colid[LF_TABLE_LINK_SUBPORT_1]].dbd_int;
    element_label2 = ldp[i][colid[LF_TABLE_LINK_PORT_2]].dbd_int;
    port2 = ldp[i][colid[LF_TABLE_LINK_SUBPORT_2]].dbd_int;

    /*
     * Make sure this link really refers to something, skip if not.
     */
    if (ldp[i][colid[LF_TABLE_LINK_NAME_1]].dbd_str == NULL
	|| ldp[i][colid[LF_TABLE_LINK_NAME_1]].dbd_str[0] == '\0'
        || ldp[i][colid[LF_TABLE_LINK_NAME_2]].dbd_str == NULL
	|| ldp[i][colid[LF_TABLE_LINK_NAME_2]].dbd_str[0] == '\0') {
      continue;
    }

    n1 = lf_find_phys_node(fp,
                      ldp[i][colid[LF_TABLE_LINK_NAME_1]].dbd_str,
                      ldp[i][colid[LF_TABLE_LINK_SLOT_1]].dbd_int,
                      element_label1);
    if (n1 == NULL) {
      LF_ERROR(("No such link1: %s/%d/%d",
                 ldp[i][colid[LF_TABLE_LINK_NAME_1]].dbd_str,
		 ldp[i][colid[LF_TABLE_LINK_SLOT_1]].dbd_int,
		 element_label1));
    }
          

    n2 = lf_find_phys_node(fp,
                      ldp[i][colid[LF_TABLE_LINK_NAME_2]].dbd_str,
                      ldp[i][colid[LF_TABLE_LINK_SLOT_2]].dbd_int,
                      element_label2);
    if (n2 == NULL) {
      LF_ERROR(("No such link2: %s/%d/%d",
                 ldp[i][colid[LF_TABLE_LINK_NAME_2]].dbd_str,
		 ldp[i][colid[LF_TABLE_LINK_SLOT_2]].dbd_int,
		 element_label2));
    }

    /*
     * If port is -1, then all ports of these two nodes
     * are linked together.  A quad cable is represented this way.
     */
    if (port1 == -1) {
      int port;
      int nport;

      if (port2 != -1) {
	LF_ERROR(("Illegal use of port -1 with %s", lf_node_string(n1, 0)));
      }

      /* only linecard xcvrs have ports */
      if (n1->ln_type == LF_NODE_LC_XCVR || n1->ln_type == LF_NODE_NIC_XCVR) {
        nport = LF_XCVR(n1)->num_conns;
      } else if (n1->ln_type == LF_NODE_XBAR) {
        nport = LF_XBAR(n1)->num_ports;
      } else { 	/* LF_NODE_NIC */
        nport = LF_NIC(n1)->num_ports;
      }
      
      /* make connections for each port */
      for (port=0; port<nport; ++port) {

	/* make sure this port not already connected */
	onp = lf_get_phys_link(n1, port, &oport);
	if (onp != NULL) {
	  LF_ERROR(("%s already connected to %s, cannot connect to %s",
		lf_node_string(n1, port),
		lf_node_string(onp, oport),
		lf_node_string(n2, port)));
	}

	/* find the topological node behind each xcvr */
	topo1 = lf_get_topo_node(n1, port, &tport1);
	if (topo1 == NULL) {
	  LF_ERROR(("No topo1, link %s/%d/%d",
                 ldp[i][colid[LF_TABLE_LINK_NAME_1]].dbd_str,
		 ldp[i][colid[LF_TABLE_LINK_SLOT_1]].dbd_int,
		 element_label2));
	}

	/* make sure this port not already connected */
	onp = lf_get_phys_link(n2, port, &oport);
	if (onp != NULL) {
	  LF_ERROR(("%s already connected to %s, cannot connect to %s",
		lf_node_string(n2, port),
		lf_node_string(onp, oport),
		lf_node_string(n1, port)));
	}

	topo2 = lf_get_topo_node(n2, port, &tport2);
	if (topo2 == NULL) {
	  LF_ERROR(("No topo2, link %s/%d/%d",
                 ldp[i][colid[LF_TABLE_LINK_NAME_2]].dbd_str,
		 ldp[i][colid[LF_TABLE_LINK_SLOT_2]].dbd_int,
		 element_label2));
	}

	/* make the links */
	lf_make_phys_link(n1, port, n2, port);
	lf_make_phys_link(n2, port, n1, port);
	lf_make_topo_link(topo1, tport1, topo2, tport2);
	lf_make_topo_link(topo2, tport2, topo1, tport1);
      }

    /* handle just one port */
    } else {

      /* make sure this port not already connected */
      onp = lf_get_phys_link(n1, port1, &oport);
      if (onp != NULL) {
	LF_ERROR(("%s already connected to %s, cannot connect to %s",
	      lf_node_string(n1, port1),
	      lf_node_string(onp, oport),
	      lf_node_string(n2, port2)));
      }

      /* find the topological node behind each xcvr */
      topo1 = lf_get_topo_node(n1, port1, &tport1);
      if (topo1 == NULL) {
	LF_ERROR(("No topo1, link %s/%d/%d",
	       ldp[i][colid[LF_TABLE_LINK_NAME_1]].dbd_str,
	       ldp[i][colid[LF_TABLE_LINK_SLOT_1]].dbd_int,
	       element_label1));
      }

      /* make sure this port not already connected */
      onp = lf_get_phys_link(n2, port2, &oport);
      if (onp != NULL) {
	LF_ERROR(("%s already connected to %s, cannot connect to %s",
	      lf_node_string(n2, port2),
	      lf_node_string(onp, oport),
	      lf_node_string(n1, port1)));
      }

      topo2 = lf_get_topo_node(n2, port2, &tport2);
      if (topo2 == NULL) {
	LF_ERROR(("No topo2, link %s/%d/%d",
	       ldp[i][colid[LF_TABLE_LINK_NAME_2]].dbd_str,
	       ldp[i][colid[LF_TABLE_LINK_SLOT_2]].dbd_int,
	       element_label2));
      }

      /* make the links */
      lf_make_phys_link(n1, port1, n2, port2);
      lf_make_phys_link(n2, port2, n1, port1);
      lf_make_topo_link(topo1, tport1, topo2, tport2);
      lf_make_topo_link(topo2, tport2, topo1, tport1);
    }
  }

  /* successful return */
  goto cleanup;
           
 except:
  retval = -1;
 cleanup:
  if (ldp != NULL) db_free_query_res(tp, ldp, nl);
  return retval;
}

/*
 * Find a physical link given a name, slot, and port
 * This is usually a transceiver, but will sometimes be a NIC or xbar, as in the
 * case of the XM ethernet linecards.
 */
lf_node_t *
lf_find_phys_node(
  lf_fabric_t *fp,
  char *name,
  int slot_label,
  int element_label)
{
  int e;
  int h;

  /* first, check for an enclosure named "name" */
  for (e=0; e<fp->num_enclosures; ++e) {
    lf_enclosure_t *ep;

    ep = fp->enclosures[e];

    /* if this is it, go to the matching line card and return the xcvr/xbar */
    if (strcmp(ep->name, name) == 0) {
      lf_linecard_t *lp;
      int port;
      int slot;

      /* validate the slot label */
      slot = slot_label - ep->lc_slotbase;
      if (slot >= ep->num_lc_slots) return NULL;

      lp = ep->slots[slot];
      if (lp == NULL) return NULL;

      /* if the card has transceivers, try those first */
      for (port=0; port<lp->num_xcvrs; ++port) {
	/* find the right port given the label */
	if (lp->xcvr_labels[port] == element_label) {
	  return lp->xcvrs[port];
	}
      }

      /* now try an xbar - element_label is xbar index */
      if (element_label < lp->num_xbars) {
        struct lf_xbar *xbp;

	xbp = LF_XBAR(lp->xbars[element_label]);
	return LF_NODE(xbp);
      }

      return NULL;	/* no match found */
    }
  }

  /* OK, not an enclosure name, try the hosts list */
  for (h=0; h<fp->num_hosts; ++h) {
    lf_host_t *hp;

    hp = fp->hosts[h];

    /* if this is it, go to the matching line card and return the xcvr */
    if (strcmp(hp->hostname, name) == 0) {
      lf_nic_t *np;
      int ni;

      for (ni=0; ni<hp->num_nics; ++ni) {
	if (hp->nics[ni]->host_nic_id == slot_label) {
	  break;
	}
      }
      if (ni >= hp->num_nics) return NULL;

      np = hp->nics[ni];
      if (element_label >= np->num_ports) return NULL;

      /* If phys_ports is filled in, use that.  else this is an XM-style device
       * with no xcvrs, so use the NIC itself.
       */
      if (np->phys_ports[element_label] != NULL) {
	return np->phys_ports[element_label];
      } else {
        return LF_NODE(np);
      }
    }
  }

  LF_ERROR(("Transceiver %s:%d:%d not found.", name, slot_label, element_label));

 except:
  return NULL;
}

/*
 * Get the first topological node behind this node
 */
lf_node_t *
lf_get_topo_node(
  lf_node_t *np,
  int port,
  int *tport)
{
  /* linecard xcvr */
  if (np->ln_type == LF_NODE_LC_XCVR || np->ln_type == LF_NODE_NIC_XCVR) {
    int p;

    p = port + LF_XCVR(np)->num_conns;
    *tport = LF_XCVR(np)->rports[p];
    return LF_XCVR(np)->ports[p];

  /* all other node types are identically topological nodes */
  } else {
    *tport = port;
    return np;
  }
}


/*
 * make a physical link between two nodes
 */
void
lf_make_phys_link(
  lf_node_t *from,
  int fport,
  lf_node_t *to,
  int tport)
{
  switch (from->ln_type) {

  case LF_NODE_LC_XCVR:
  case LF_NODE_NIC_XCVR:
    LF_XCVR(from)->ports[fport] = to;
    LF_XCVR(from)->rports[fport] = tport;
    break;

  case LF_NODE_XBAR:
    LF_XBAR(from)->phys_ports[fport] = to;
    LF_XBAR(from)->phys_rports[fport] = tport;
    break;

  case LF_NODE_NIC:
    LF_NIC(from)->phys_ports[fport] = to;
    LF_NIC(from)->phys_rports[fport] = tport;
    break;
  }
}

/*
 * make a topological link between two nodes
 */
void
lf_make_topo_link(
  lf_node_t *from,
  int fport,
  lf_node_t *to,
  int tport)
{
  switch (from->ln_type) {

  case LF_NODE_XBAR:
    LF_XBAR(from)->topo_ports[fport] = to;
    LF_XBAR(from)->topo_rports[fport] = tport;
    break;

  case LF_NODE_NIC:
    LF_NIC(from)->topo_ports[fport] = to;
    LF_NIC(from)->topo_rports[fport] = tport;
    break;

  case LF_NODE_LC_XCVR:
  case LF_NODE_NIC_XCVR:
    break;
  }
}

/*
 * get the physical link for a node/port
 */
union lf_node *
lf_get_phys_link(
  lf_node_t *np,
  int port,
  int *oport)
{
  switch (np->ln_type) {

  case LF_NODE_XBAR:
    if (oport != NULL) {
      *oport = LF_XBAR(np)->phys_rports[port];
    }
    return LF_XBAR(np)->phys_ports[port];

  case LF_NODE_NIC:
    if (oport != NULL) {
      *oport = LF_NIC(np)->phys_rports[port];
    }
    return LF_NIC(np)->phys_ports[port];

  case LF_NODE_LC_XCVR:
  case LF_NODE_NIC_XCVR:
    if (oport != NULL) {
      *oport = LF_XCVR(np)->rports[port];
    }
    return LF_XCVR(np)->ports[port];
    break;
  }
  return NULL;
}

/*
 * get the topological link for a node/port
 */
union lf_node *
lf_get_topo_link(
  lf_node_t *np,
  int port,
  int *oport)
{
  switch (np->ln_type) {

  case LF_NODE_XBAR:
    if (oport != NULL) {
      *oport = LF_XBAR(np)->topo_rports[port];
    }
    return LF_XBAR(np)->topo_ports[port];

  case LF_NODE_NIC:
    if (oport != NULL) {
      *oport = LF_NIC(np)->topo_rports[port];
    }
    return LF_NIC(np)->topo_ports[port];

  case LF_NODE_LC_XCVR:
  case LF_NODE_NIC_XCVR:
    break;
  }
  return NULL;
}

/*
 * create implicit links for an enclosure
 */
void
lf_create_implicit_links(
  struct lf_enclosure *ep)
{
  int s;

  /* make all implicit links for all slots */
  for (s=0; s<ep->num_slots; ++s) {
    struct lf_linecard *lp;

    lp = ep->slots[s];
    if (lp != NULL) {
      lf_make_linecard_internal_links(lp);
      lf_make_linecard_edge_links(lp);
    }
  }
}

/*
 * make the internal links for a linecard
 */
void
lf_make_linecard_internal_links(
  lf_linecard_t *lp)
{
  int x;

  /* loop over each xbar */
  for (x=0; x<lp->num_xbars; ++x) {
    lf_node_t *xp;
    int p;

    xp = lp->xbars[x];

    /* check each connection for a link */
    for (p=0; p<LF_XBAR(xp)->num_ports; ++p) {
      struct lf_linecard_xbar_conn *cp;

      /* if these is a connection, make the link */
      cp = lp->def->xbar_conn[x][p];
      if (cp != NULL) {
	int xcvr_port;

	/* convert xcvr port to internal link index (see lf_fabric.c) */
	xcvr_port = cp->port + LF_XCVR(lp->xcvrs[cp->index])->num_conns;
	
	/* make the links */
	lf_make_phys_link(xp, p, lp->xcvrs[cp->index], xcvr_port);
	lf_make_phys_link(lp->xcvrs[cp->index], xcvr_port, xp, p);
      }
    }
  }
}

/*
 * make the edge links for a linecard
 */
static void
lf_make_linecard_edge_links(
  lf_linecard_t *my_lp)
{
  struct lf_enclosure_def *edp;
  struct lf_linecard *his_lp;
  struct lf_linecard_def *my_dp;
  struct lf_linecard_def *his_dp;
  struct lf_enclosure *ep;
  int my_pin;

  /* get pointers to definition structures */
  ep = my_lp->enclosure;
  edp = ep->def;
  my_dp = my_lp->def;

  /*
   * loop through every pin i have and see where it goes on my side
   * and what it's connected to through the backplane
   */
  for (my_pin=0; my_pin<my_dp->num_pins; ++my_pin) {
    union lf_node *my_node;
    union lf_node *his_node;
    struct lf_linecard_pin_conn *my_pin_conn;
    struct lf_linecard_pin_conn *his_pin_conn;
    struct lf_slot_conn *slot_conn;
    int my_port;
    int his_port;
    int his_pin;
    int his_slot;

    /* slot connection information for my slot */
    slot_conn = edp->conn[my_lp->slot];

    /* find the partner */
    his_slot = slot_conn[my_pin].slot;
    his_pin = slot_conn[my_pin].pin;
    his_lp = ep->slots[his_slot];

    /* skip if no card in that slot */
    if (his_lp == NULL) continue;

    his_dp = his_lp->def;

    /* get my node and port */
    my_pin_conn = &my_dp->pin_conn[my_pin];
    my_port = my_pin_conn->port;
    if (my_pin_conn->is_xbar) {
      my_node = my_lp->xbars[my_pin_conn->index];
      if (LF_XBAR(my_node)->phys_ports[my_port] != NULL) continue;
      
    } else {
      my_node = my_lp->xcvrs[my_pin_conn->index];

      /* convert xcvr port to internal link index (see lf_fabric.c) */
      my_port += LF_XCVR(my_node)->num_conns;

      if (LF_XCVR(my_node)->ports[my_port] != NULL) continue;
    }

    /* get his node and port */
    his_pin_conn = &his_dp->pin_conn[his_pin];
    his_port = his_pin_conn->port;
    if (his_pin_conn->is_xbar) {
      his_node = his_lp->xbars[his_pin_conn->index];
    } else {
      /* convert xcvr port to internal link index (see lf_fabric.c) */
      his_node = his_lp->xcvrs[his_pin_conn->index];
      his_port += LF_XCVR(his_node)->num_conns;
    }

    /* connect the two nodes */
    lf_make_phys_link(my_node, my_port, his_node, his_port);
    lf_make_phys_link(his_node, his_port, my_node, my_port);

    /* if both xbars, make the topo link also */
    if (my_node->ln_type == LF_NODE_XBAR
        && his_node->ln_type == LF_NODE_XBAR) {
      lf_make_topo_link(my_node, my_port, his_node, his_port);
      lf_make_topo_link(his_node, his_port, my_node, my_port);
    }
  }
}

/*
 * Allocate a linecard for a slot in an enclosure
 */
lf_linecard_t *
lf_allocate_linecard(
  struct lf_enclosure *ep,
  int slot,
  char *product_id,
  char *serial_no)
{
  lf_linecard_t *lp;

  LF_CALLOC(lp, lf_linecard_t, 1);

  /* save in ep */
  ep->slots[slot] = lp;

  /* fill in what we know about it */
  lp->slot = slot;
  lp->backplane = (slot >= ep->num_lc_slots);
  lp->enclosure = ep;
  LF_DUP_STRING(lp->product_id, product_id);
  LF_DUP_STRING(lp->serial_no, serial_no);
  LF_CALLOC(lp->data, struct lf_linecard_data, 1);

  /* load definitions for the rest of it */
  if (lf_fill_linecard(lp) == -1) {
    LF_ERROR(("Filling in linecard info"));
  }

  /* make the implicit internal links for this card */
  lf_make_linecard_internal_links(lp);		/* within the card */
  lf_make_linecard_edge_links(lp);		/* within the switch */
  
  return lp;
 except:
  return NULL;
}

/*
 * Allocate a structure for an enclosure and fill it in
 */
lf_enclosure_t *
lf_allocate_enclosure(
  char *product_id,
  char *name)
{
  lf_enclosure_t *ep;
  int rc;

  LF_CALLOC(ep, lf_enclosure_t, 1);

  /* fill in the easy stuff */
  LF_DUP_STRING(ep->product_id, product_id);
  LF_DUP_STRING(ep->name, name);
  LF_CALLOC(ep->data, struct lf_enclosure_data, 1);

  /* fill in implicit information from product ID files */
  rc = lf_fill_enclosure(ep);
  if (rc == -1) goto except;

  return ep;

 except:
  return NULL;
}

/*
 * close up the database tables
 */
void
lf_close_fabric_db(
  struct lf_fabric_db *fdp)
{
  if (fdp->host_tbl != NULL) db_close_table(fdp->host_tbl);
  if (fdp->nic_tbl != NULL) db_close_table(fdp->nic_tbl);
  if (fdp->enclosure_tbl != NULL) db_close_table(fdp->enclosure_tbl);
  if (fdp->linecard_tbl != NULL) db_close_table(fdp->linecard_tbl);
  if (fdp->link_tbl != NULL) db_close_table(fdp->link_tbl);

  db_close_database(fdp->database);
}

/*
 * Allocate a struct for holding fabric DB information.  This allows the
 * struct to be opaque and used as a handle.
 */
struct lf_fabric_db *
lf_fabric_db_handle(db_database_ptr_t dp)
{
  struct lf_fabric_db *fdp;

  LF_CALLOC(fdp, struct lf_fabric_db, 1);
  fdp->database = dp;
  return fdp;

 except:
  return NULL;
}
